// Vérifier si on est sur la page admin
const isAdminPage = window.location.pathname.includes('admin.html');

// Fonction de déconnexion
function logout() {
    sessionStorage.removeItem('adminLoggedIn');
    window.location.href = 'login.html';
}

const postsContainer = document.getElementById('postsContainer');
const addPostBtn = document.getElementById('addPostBtn');
const postModal = document.getElementById('postModal');
const closeModal = document.getElementById('closeModal');
const postForm = document.getElementById('postForm');

// Charger les posts au démarrage
loadPosts();

// Gestion du modal d'image (lightbox)
const imageModal = document.getElementById('imageModal');
const modalImage = document.getElementById('modalImage');
const closeImageModal = document.getElementById('closeImageModal');

function openImageModal(imageSrc, imageAlt) {
    if (!imageModal || !modalImage) return;
    
    modalImage.src = imageSrc;
    modalImage.alt = imageAlt;
    imageModal.classList.add('active');
    document.body.style.overflow = 'hidden';
}

if (closeImageModal) {
    closeImageModal.addEventListener('click', (e) => {
        e.stopPropagation();
        imageModal.classList.remove('active');
        document.body.style.overflow = '';
    });
}

if (imageModal) {
    imageModal.addEventListener('click', (e) => {
        if (e.target === imageModal) {
            imageModal.classList.remove('active');
            document.body.style.overflow = '';
        }
    });
}

document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape' && imageModal && imageModal.classList.contains('active')) {
        imageModal.classList.remove('active');
        document.body.style.overflow = '';
    }
});

async function loadPosts() {
    try {
        const response = await fetch('posts.json?t=' + Date.now());
        const posts = await response.json();
        
        postsContainer.innerHTML = '';
        
        if (posts.length === 0) {
            postsContainer.innerHTML = `
                <div style="text-align: center; padding: 3rem 1rem; color: #e1e1e1; opacity: 0.8;">
                    <img src="kittycat.png" 
                         alt="Chat sur ordinateur" 
                         style="max-width: 400px; width: 100%; margin-bottom: 1.5rem;">
                    <p style="font-size: 1.2rem; color: #ec9d68;">Rien à voir ici pour le moment...</p>
                    ${isAdminPage ? '<p style="font-size: 0.9rem; margin-top: 0.5rem;">Cliquez sur "+ Ajouter un article" pour commencer !</p>' : ''}
                </div>
            `;
        } else {
            posts.forEach(post => {
                const article = createPostElement(post);
                postsContainer.appendChild(article);
            });
        }
    } catch (error) {
        console.error('Erreur lors du chargement des posts:', error);
        postsContainer.innerHTML = `
            <article class="post">
                <div class="post-header">
                    <h2 class="post-title">Erreur</h2>
                    <div class="post-meta">Erreur de chargement</div>
                </div>
                <div class="post-content">
                    <p>Impossible de charger les articles. Assurez-vous que le fichier posts.json existe.</p>
                </div>
            </article>
        `;
    }
}

function createPostElement(post) {
    const article = document.createElement('article');
    article.className = 'post';
    
    const imageHtml = post.image ? `
        <img src="${post.image}" alt="${escapeHtml(post.title)}" class="post-image" onclick="openImageModal('${post.image}', '${escapeHtml(post.title)}')">
        <p class="image-hint">Vous pouvez cliquer sur l'image pour la voir en entier :)</p>
    ` : '';
    
    article.innerHTML = `
        <div class="post-header">
            <h2 class="post-title">${escapeHtml(post.title)}</h2>
            <div class="post-meta">Publié le ${post.date}</div>
        </div>
        ${imageHtml}
        <div class="post-content">
            <p>${escapeHtml(post.content)}</p>
        </div>
        ${isAdminPage ? `<button class="delete-btn" data-post-id="${post.id}">Supprimer</button>` : ''}
    `;
    
    if (isAdminPage) {
        const deleteBtn = article.querySelector('.delete-btn');
        deleteBtn.addEventListener('click', () => deletePost(post.id));
    }
    
    return article;
}

async function savePost(post) {
    try {
        const response = await fetch('posts.json?t=' + Date.now());
        const posts = await response.json();
        
        posts.unshift(post);
        
        const saveResponse = await fetch('save_posts.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(posts)
        });
        
        const result = await saveResponse.json();
        
        if (result.success) {
            setTimeout(() => loadPosts(), 300);
        } else {
            alert('Erreur: ' + result.message);
        }
    } catch (error) {
        console.error('Erreur lors de la sauvegarde:', error);
        alert('Erreur lors de la sauvegarde du post');
    }
}

async function deletePost(postId) {
    if (confirm('Êtes-vous sûr de vouloir supprimer cet article ?')) {
        try {
            const response = await fetch('posts.json?t=' + Date.now());
            const posts = await response.json();
            
            const filteredPosts = posts.filter(post => post.id !== postId);
            
            const saveResponse = await fetch('save_posts.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(filteredPosts)
            });
            
            const result = await saveResponse.json();
            
            if (result.success) {
                setTimeout(() => loadPosts(), 300);
            } else {
                alert('Erreur: ' + result.message);
            }
        } catch (error) {
            console.error('Erreur lors de la suppression:', error);
            alert('Erreur lors de la suppression du post');
        }
    }
}

if (addPostBtn) {
    addPostBtn.addEventListener('click', () => {
        postModal.classList.add('active');
    });
}

if (closeModal) {
    closeModal.addEventListener('click', () => {
        postModal.classList.remove('active');
    });
}

if (postModal) {
    postModal.addEventListener('click', (e) => {
        if (e.target === postModal) {
            postModal.classList.remove('active');
        }
    });
}

if (postForm) {
    postForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const title = document.getElementById('postTitle').value;
        const content = document.getElementById('postContent').value;
        const imageInput = document.getElementById('postImage');

        const date = new Date().toLocaleDateString('fr-FR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });

        const post = {
            id: Date.now(),
            title,
            content,
            date
        };

        if (imageInput.files && imageInput.files[0]) {
            const file = imageInput.files[0];
            const img = new Image();
            const reader = new FileReader();
            
            reader.onload = function(e) {
                img.src = e.target.result;
            };
            
            img.onload = async function() {
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');
                
                const maxWidth = 1200;
                let width = img.width;
                let height = img.height;
                
                if (width > maxWidth) {
                    height = (height * maxWidth) / width;
                    width = maxWidth;
                }
                
                canvas.width = width;
                canvas.height = height;
                ctx.drawImage(img, 0, 0, width, height);
                post.image = canvas.toDataURL('image/jpeg', 0.8);
                
                await savePost(post);
                postForm.reset();
                postModal.classList.remove('active');
            };
            
            reader.readAsDataURL(file);
        } else {
            await savePost(post);
            postForm.reset();
            postModal.classList.remove('active');
        }
    });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}